<?php

namespace App\Repositories\General;

use App\Events\ConversationEvent;
use Exception;
use Throwable;
use App\Models\User;
use App\Models\Message;
use App\Models\Conversation;
use Illuminate\Http\Request;
use App\Services\ImageService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ConversationRepository
{
    protected $imageService;

    public function __construct(ImageService $imageService)
    {
        $this->imageService = $imageService;
    }

    public function getAll()
    {
        $conversations = auth()->user()->conversations()->with([
            'participants.images',
            'messages.user',
            'messages.images',
        ])
            ->has('messages')
            ->latest('updated_at')
            ->paginate(config('pagination.general'))
            ->withQueryString();

        return compact('conversations');
    }

    protected function getOldConversation(User $user)
    {
        $loginUser = auth()->user();
        $oldConversations = $loginUser->conversations()->with([
            'participants.images',
        ])->get();
        $conversation = null;
        $conversationList = [];
        if ($oldConversations->count() > 0) {
            foreach ($oldConversations as $oldConversation) {
                $result = $oldConversation->participants->filter(function ($participant) use ($user) {
                    return $participant->id == $user->id;
                });

                if (count($result) > 0) {
                    $conversationList[] = $oldConversation;
                }
            }
        }
        if (count($conversationList) > 0) {
            foreach ($conversationList as $filteredConversation) {
                $participantCount = $filteredConversation->participants->count();

                if ($participantCount > 0) {
                    $conversation = $filteredConversation;
                    break;
                }
            }
        }
        if (!$conversation) {
            $conversation = Conversation::create([]);
            $conversation->participants()->saveMany([$loginUser, $user]);
        }

        return $conversation;
    }

    protected function getOldGroupConversation(Conversation $conversation)
    {
    }

    public function show(User $user)
    {
        $conversation = $this->getOldConversation($user);
        $messages = $conversation?->messages()->with([
            'images',
            'user.images',
            'seenUsers.images',
        ])
            ->latest('created_at')
            ->paginate(config('pagination.general'))
            ->withQueryString();

        return compact('conversation', 'messages');
    }

    public function save(Request $request)
    {
        try {
            DB::beginTransaction();
            $user = User::findOrFail($request->user_id);

            $conversation = $this->getOldConversation($user);

            $loginUser = auth()->user();
            if (!$conversation) {
                $conversation = Conversation::create([]);
                $conversation->participants()->saveMany([$loginUser, $user]);
            }
            $message = $conversation->messages()->updateOrCreate([
                'id' => $request->message_id,
            ], [
                'user_id' => $loginUser->id,
                'content' => $request->content,
            ]);

            if ($request->images && count($request->images) > 0) {
                $images = $this->imageService->uploadMany($request->images, Message::class, $message->id);
                $message->images()->saveMany($images);
            }

            $message = Message::with([
                'images',
                'user.images',
                'seenUsers.images'
            ])->find($message->id);

            broadcast(new ConversationEvent($conversation, $message));

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Message saved failed.');
        }
    }

    public function seen(User $user, $messageId)
    {
        $conversation = $this->getOldConversation($user);

        abort_if($conversation == null, 404, 'Not Found.');
        if ($conversation) {
            $message = $conversation->messages()->findOrFail($messageId);
        }

        $message->seenUsers()->sync([auth()->id()]);
    }

    public function deleteMessage(User $user, $messageId)
    {
        $conversation = $this->getOldConversation($user);

        abort_if($conversation == null, 404, 'Not Found.');
        if ($conversation) {
            $message = $conversation->messages()->findOrFail($messageId);
            $message->delete();
        }
    }

    public function saveGroup(Request $request)
    {
        try {
            DB::beginTransaction();
            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Conversation Group saved failed.');
        }
    }

    public function showGroup($conversationId)
    {
    }

    public function saveGroupMessage(Request $request, $conversationId)
    {
    }

    public function seenGroup($conversationId, $messageId)
    {
    }

    public function leaveGroup($conversationId)
    {
    }

    public function deleteGroupMessage($conversationId, $messageId)
    {
    }

    public function deleteConversation($conversationId)
    {
        $loginUserId = auth()->id();
        $conversation = Conversation::findOrFail($conversationId);

        if ($conversation->participants->count > 2) {
            abort_unless($conversation->user_id == $loginUserId, 403, 'Unauthorized action.');
        } else {
            abort_unless($conversation->participants()->find($loginUserId)?->exists(), 403, 'Unauthorized action.');
        }
        $conversation->delete();
    }
}
