<?php

namespace App\Models;

use App\Enums\PostType;
use App\Enums\GroupType;
use App\Enums\DeletedType;
use App\Enums\GroupRequester;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Group extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'category_id',
        'name',
        'description',
        'type',
        'deleted_type',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'type' => GroupType::class,
        'deleted_type' => DeletedType::class,
    ];

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function members(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'group_members')->withPivot(['is_seen', 'is_accepted', 'requester'])->withTimestamps();
    }

    public function unacceptedMembers(): BelongsToMany
    {
        return $this->members()->wherePivot('is_accepted', false);
    }

    public function unacceptedMembersByGroupOwner(): BelongsToMany
    {
        return $this->members()->wherePivot('is_accepted', false)->wherePivot('requester', GroupRequester::GROUP_OWNER->value);
    }

    public function unacceptedMembersByUser(): BelongsToMany
    {
        return $this->members()->wherePivot('is_accepted', false)->wherePivot('requester', GroupRequester::USER->value);
    }

    public function acceptedMembers(): BelongsToMany
    {
        return $this->members()->wherePivot('is_accepted', true);
    }

    public function posts(): HasMany
    {
        return $this->hasMany(Post::class, 'origin_id')->where('posts.type', PostType::GROUP);
    }

    public function unacceptedPosts(): HasMany
    {
        return $this->posts()->accepted(false);
    }

    public function acceptedPosts(): HasMany
    {
        return $this->posts()->accepted(true);
    }

    public function images(): MorphMany
    {
        return $this->morphMany(Image::class, 'imageable');
    }

    public function scopeOwned($query)
    {
        $query->where('user_id', auth()->id());
    }

    public function scopePublic($query)
    {
        $query->where('type', GroupType::PUBLIC);
    }

    public function scopeProtected($query)
    {
        $query->where('type', GroupType::PROTECTED);
    }

    public function scopeFilter($query, $filters)
    {
        $query->when($filters['search'] ?? false, function ($query, $searchText) {
            $query->whereHas('category', function ($query) use ($searchText) {
                $query->where('name', 'like', '%' . $searchText . '%');
            })->orWhere('name', 'like', '%' . $searchText . '%');
        });
    }

    public function isPublic()
    {
        return $this->type == GroupType::PUBLIC;
    }

    public function isProtected()
    {
        return $this->type == GroupType::PROTECTED;
    }
}
